from dataclasses import dataclass, asdict

from django.core.handlers.wsgi import WSGIRequest
from django.http import HttpResponse
from django.shortcuts import redirect
from django_tables2 import SingleTableView, RequestConfig, Column

from .forms import WordFilterFormHelper
from .filters import WordFilter
from .models import Word, Practice, Synonym, ExampleUsage
from .tables import WordTable, PracticeTable
from . import logger


@dataclass
class Session:
    """
    comfortable container for the session dict
    """
    practice_list: list
    active: bool
    subset: str  # restrict the filter to words starting with subset
    synonyms: bool
    examples: bool


class PagedFilteredTableView(SingleTableView):
    """
    Class to incorporate django-tables, crispy and django-filter
    """
    filter_class = None
    formhelper_class = None
    context_filter_name = 'filter'
    practice_table_class = None

    def get_queryset(self, **kwargs):
        """
        get the initial query set ordered, and fix the crispy form and the filters
        :param kwargs:
        :return: a query set containing all words. filters can restrict the
        selection, see filters.py.
        """
        qs = super().get_queryset().order_by('source_language', 'source_text')
        if len(self.session.subset) == 1:
            qs = qs.filter(source_text__istartswith=self.session.subset)
        self.filter = self.filter_class(self.request.GET, queryset=qs)
        self.filter.form.helper = self.formhelper_class(
            user=self.request.user,
            practice=self.session.active)
        return self.filter.qs

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.ip = None
        self.session = None

    def get_table(self, **kwargs):
        """
        get the table and return a paginated table.
        :param kwargs:
        :return:
        """
        columns = list()
        if self.session.synonyms:
            columns.append(('synonym', Column()))

        if self.session.examples:
            columns.append(('example', Column()))
        table = self.table_class(self.get_table_data(), extra_columns=columns,
                                 issuper=not self.request.user.is_anonymous)
        RequestConfig(self.request,
                      paginate={'per_page': self.paginate_by}).configure(table)
        return table

    def get_context_data(self, **kwargs):
        """
        add the filter class to the context. used by the html template.
        :param kwargs:
        :return:
        """
        context = super().get_context_data()
        if self.session.active:
            context['practice'] = PracticeTable(data=
            Word.objects.filter(id__in=self.session.practice_list).order_by(
                "source_text"))
        context[self.context_filter_name] = self.filter
        return context

    def get(self, request: WSGIRequest, *args, **kwargs) -> HttpResponse:
        """
        Override default get to process filter and other submit buttons
        intercept any special actions to presentation of the word list.
        :param request:
        :return: HttpResponse
        """

        self.ip = request.META.get("REMOTE_ADDR")  # this ip is used to store
        # temporary practice words
        # inherit previous session if present
        startsession = {'practice_list': [],
                        'active': False,
                        'subset': "a",
                        'synonyms': False,
                        'examples': False}
        try:
            self.session = Session(**request.session.get('session', startsession))
        except TypeError:
            self.session = Session(**startsession)
        self.session.practice_list = list(Practice.objects.filter(
            ip=self.ip).values_list(
            "word_id", flat=True).order_by("word__source_text"))
        logger.debug(self.session)
        if any([let in request.GET for let in 'abcdefghijklmnopqrstuvwxyz']):
            for let in request.GET:
                if len(let) == 1:  # really pressed a letter button?
                    logger.info(f"set subset selection to {let}")
                    self.session.subset = let

        if 'practice' in request.GET:  # can be reset or practice
            if self.session.active:
                # reset practice table
                self.session.practice_list = []
                self.delete_practice_words()
                self.session.active = False
                logger.info(f'reset pressed {self.session}')
            else:
                practice_list = set(self.session.practice_list)
                practice_list.update(set(self.load_practice_words()))
                self.practice_table = PracticeTable(
                    data=Word.objects.filter(id__in=practice_list).order_by(
                        "source_text"))
                self.session.active = True
                logger.info(f'practice pressed {self.session}, {practice_list}')
        if "synonyms" in request.GET:
            self.session.synonyms = not self.session.synonyms
            logger.info(f"synonyms in request")
            # add synonyms column to table
        if "addword" in request.GET:
            logger.info("editword in request")
            return redirect("admin/elo/word")
            # go to admin/Word
        if "example" in request.GET:
            self.session.examples = not self.session.examples
            logger.info("example in request")
            # add example column to table

        if self.session.active:
            self.make_collection_changes(self.request.GET)

        logger.debug(asdict(self.session))

        self.request.session['session'] = asdict(self.session)  # keep session data
        return super().get(request, *args, **kwargs)

    def delete_practice_words(self):
        pass

    def make_collection_changes(self, g: dict):
        pass

    def get_absolute_url(self):
        # TODO: replace this with the word translate form
        return f'admin/elo/word/{self.id}'

    def load_practice_words(self) -> list:
        pass


class WordTableView(PagedFilteredTableView):
    """
    Sub class to display words
    """
    model = Word
    table_class = WordTable
    practice_table_class = PracticeTable
    template_name = 'elo/word.html'
    paginate_by = 50
    filter_class = WordFilter
    formhelper_class = WordFilterFormHelper

    def keep_practice_words(self):
        """Store practice words in database"""
        # first remove deleted words
        Practice.objects.exclude(word_id__in=self.session.practice_list).delete()
        # add records one by one from the include list
        for word in self.session.practice_list:
            if not Practice.objects.filter(word_id=word, ip=self.ip).exists():
                practice = Practice(word_id=word, ip=self.ip)
                practice.save()

    def load_practice_words(self) -> list:
        """load all practice words belonging to the current ip"""
        return Practice.objects.filter(
            ip=self.ip).values_list("word_id", flat=True)

    def delete_practice_words(self):
        Practice.objects.filter(ip=self.ip).delete()

    def make_collection_changes(self, g: dict):
        """
        make changes to the practice table based on hidden input practice_values
        :return: None
        """
        words = g.get('practice_values', '').split(',')
        logger.info(words)
        try:
            words = list(map(int, words))
        except ValueError:
            words = list()
        words_add = {w for w in words if w > 0}
        words_remove = {-w for w in words if w < 0}
        if self.session.active:
            practice_set = set(self.session.practice_list)
            practice_set = practice_set.union(words_add)
            practice_set = practice_set.difference(words_remove)
            self.session.practice_list = list(practice_set)  # session cannot
            # contain sets
        else:
            self.session.practice_list = list(words_add)
        self.keep_practice_words()

    def get_table_data(self):
        data = self.get_queryset().values()
        logger.debug(f'len of data={len(data)}')
        for record in data:
            if self.session.synonyms:
                record['synonym'] = ', '.join(Synonym.objects.filter(
                    word__source_text=record['source_text']).values_list(
                    'synonym', flat=True).order_by('synonym'))
            if self.session.examples:
                record['example'] = ', '.join(ExampleUsage.objects.filter(
                    word__source_text=record['source_text']).values_list(
                    'example', flat=True).order_by('example'))
        return data
