import django
from django.db import models


class Language(models.Model):
    name = models.TextField(db_column='NAME', blank=True, null=True)
    adddate = models.DateTimeField(db_column='ADDDATE', blank=True, null=True)
    mtdate = models.DateTimeField(db_column='MTDATE', blank=True, null=True)
    mtanr = models.TextField(db_column='MTANR', blank=True, null=True)
    flag_url = models.TextField(db_column='FLAG_URL', blank=True, null=True)
    comments = models.TextField(db_column='COMMENTS', blank=True, null=True)

    class Meta:
        managed = True
        db_table = 'ELOQUENT_LANGUAGE'

    def __str__(self):
        return self.name


class Dialect(models.Model):
    language = models.ForeignKey(Language,
                                 on_delete=models.CASCADE)  # Field name made lowercase. This field type is a guess.
    name = models.TextField(db_column='NAME', blank=True,
                            null=True)  # Field name made lowercase. This field type is a guess.
    adddate = models.DateTimeField(db_column='ADDDATE', blank=True,
                                   null=True)  # Field name made lowercase.
    mtdate = models.DateTimeField(db_column='MTDATE', blank=True,
                                  null=True)  # Field name made lowercase.
    mtanr = models.TextField(db_column='MTANR', blank=True,
                             null=True)  # Field name made lowercase. This field type is a guess.

    class Meta:
        managed = True
        db_table = 'ELOQUENT_DIALECT'

    def __str__(self):
        return f"{self.language}-{self.name}"


class Domain(models.Model):
    language = models.ForeignKey(Language,
                                 on_delete=models.CASCADE)  # Field name made lowercase. This field type is a guess.
    name = models.TextField(db_column='NAME', blank=True,
                            null=True)  # Field name made lowercase. This field type is a guess.
    mtdate = models.DateTimeField(db_column='MTDATE', blank=True,
                                  null=True)  # Field name made lowercase.
    mtanr = models.TextField(db_column='MTANR', blank=True,
                             null=True)  # Field name made lowercase. This field type is a guess.
    adddate = models.DateTimeField(db_column='ADDDATE', blank=True,
                                   null=True)  # Field name made lowercase.

    class Meta:
        managed = True
        db_table = 'ELOQUENT_DOMAIN'

    def __str__(self):
        return f'{self.name}'


class Mp3(models.Model):
    name = models.TextField(db_column='NAME', blank=True, null=True)
    mime_type = models.TextField(db_column='MIME_TYPE', blank=True, null=True)
    doc_size = models.TextField(db_column='DOC_SIZE', blank=True, null=True)
    dad_charset = models.TextField(db_column='DAD_CHARSET', blank=True, null=True)
    content_type = models.TextField(db_column='CONTENT_TYPE', blank=True, null=True)
    content = models.TextField(db_column='CONTENT', blank=True, null=True)
    blob_content = models.BinaryField(db_column='BLOB_CONTENT', blank=True, null=True)
    addate = models.DateTimeField(db_column='ADDATE', blank=True, null=True)
    mtdate = models.DateTimeField(db_column='MTDATE', blank=True, null=True)
    # name made lowercase.
    mtanr = models.TextField(db_column='MTANR', blank=True, null=True)

    class Meta:
        managed = True
        db_table = 'ELOQUENT_MP3'

    def __str__(self):
        return self.name


class Role(models.Model):
    name = models.CharField(db_column='NAME', max_length=32)
    adddate = models.DateTimeField(auto_now_add=True, db_column='ADDDATE')
    mtdate = models.DateTimeField(db_column='MTDATE', null=True, blank=True)
    mtanr = models.IntegerField(db_column='MTANR')

    class Meta:
        managed = True
        db_table = 'ELOQUENT_ROLE'

    def __str__(self):
        return f'role-{self.name}'


class EloquentUser(django.contrib.auth.models.User):
    anr = models.CharField(db_column='ANR', max_length=7)
    role = models.ForeignKey(Role, on_delete=models.CASCADE)
    active = models.BooleanField(db_column='ACTIVE')
    name = models.CharField(db_column='NAME', max_length=64)
    sortname = models.CharField(db_column='SORTNAME', max_length=64)
    lastlogin = models.DateTimeField(db_column='LASTLOGIN', null=True, blank=True)
    adddate = models.DateTimeField(db_column='ADDDATE', auto_now_add=True)
    mtdate = models.DateTimeField(db_column='MTDATE', null=True, blank=True)
    telno = models.CharField(db_column='TELNO', max_length=64, null=True, blank=True)
    attempt = models.IntegerField(db_column='ATTEMPT', default=0, null=True)

    class Meta:
        managed = True
        db_table = 'ELOQUENT_USER'

    def __str__(self):
        return f'user-{self.name}'


class Word(models.Model):
    source_text = models.CharField(max_length=256)
    source_language = models.ForeignKey(
        Language, verbose_name="Source language", related_name="source_language", on_delete=models.CASCADE)
    dialect = models.ForeignKey(Dialect, verbose_name="Pronunciation", on_delete=models.CASCADE)
    domain = models.ForeignKey(Domain, on_delete=models.CASCADE, default=1)
    translated_text = models.CharField(max_length=256)
    target_language = models.ForeignKey(
        Language, verbose_name="Translated language", related_name="target_language", on_delete=models.CASCADE)
    mp3_number = models.IntegerField()
    mp3 = models.FileField(upload_to='mp3', blank=True, null=True)
    adddate = models.DateTimeField(auto_now_add=True)
    mtdate = models.DateTimeField(null=True, blank=True)

    class Meta:
        managed = True
        db_table = 'ELOQUENT_WORD'

    def get_absolute_url(self):
        # TODO: replace this with the word translate form
        return f'admin/elo/word/{self.id}'

    def __str__(self):
        return f'{self.source_text}'


class Translation(models.Model):
    word = models.ForeignKey(Word, related_name="translation_word",
                             on_delete=models.CASCADE)
    lang = models.ForeignKey(Language, related_name="lang",
                             on_delete=models.CASCADE,
                             default=1)
    text = models.CharField(max_length=256, blank=True, null=True)

    class Meta:
        managed = True
        db_table = 'ELOQUENT_TRANSLATION'

    def __str__(self):
        return f'{self.word}->{self.lang}'


class ExampleUsage(models.Model):
    word = models.ForeignKey(Word, related_name="example_word",
                             on_delete=models.CASCADE)
    example = models.CharField(max_length=512, blank=True, null=True)

    class Meta:
        managed = True
        db_table = 'ELOQUENT_EXAMPLE'

    def __str__(self):
        return f'{self.word}-{self.example}'


class Synonym(models.Model):
    word = models.ForeignKey(Word, related_name="synonym_word",
                             on_delete=models.CASCADE)
    synonym = models.CharField(max_length=256, blank=True, null=True)

    class Meta:
        managed = True
        db_table = 'ELOQUENT_SYNONYM'

    def __str__(self):
        return f'{self.word}-{self.synonym}'


class Practice(models.Model):
    word = models.ForeignKey(Word, on_delete=models.CASCADE)
    ip = models.CharField(max_length=20)

    class Meta:
        managed = True
        db_table = 'PRACTICE'

    def __str__(self):
        return f"{self.ip}-{self.word}"


class WordDialect(models.Model):
    word = models.ForeignKey(Word, on_delete=models.CASCADE)
    dialect = models.ForeignKey(Dialect, on_delete=models.CASCADE)
    mp3 = models.FileField(upload_to='mp3', blank=True, null=True)

    class Meta:
        managed = True
        db_table = 'WORDDIALECT'

    def __str__(self):
        return f"{self.word}-{self.dialect}"

