import django_tables2 as tables
from django.utils.html import format_html
from django.shortcuts import reverse

from .models import Word


def helper(record):
    if isinstance(record, Word):
        return record.id
    else:
        return record['id']


class SuperMeta:
    """
    Meta class for word tables. Defines fields and styles.
    """
    template_name = "django_tables2/bootstrap4.html"
    attrs = {
        "class": "table table-sm table-hover table-striped",
        "id": "word-table",
        "thead": {
            "class": "thead-light"
        },
        "tbody": {
            "id": "word-body"
        },
        "td": {
            "class": "word-td"
        }
    }
    row_attrs = {
        "data-id": helper
    }
    model = Word
    empty_text = "No words fitting the selection criteria."
    fields = ['source_text', 'translated_text', 'mp3']
    orderable = False


class WordTable(tables.Table):
    class Meta(SuperMeta):
        pass

    def render_mp3(self, value, record):
        return format_html(
            f"""<audio controls style="width: 100px; height: 20px;"
            controlsList="nodownload">
                    <source src="media/{value}" type="audio/mpeg">
                    Your browser does not support the audio element.
                </audio>""")

    def render_source_text(self, value, record):

        return format_html(
            f"""
            <a href="#">{value}</a>"""
        )

    def __init__(self, *args, **kwargs):
        if 'issuper' in kwargs:
            self.linkify = kwargs.pop('issuper')
        else:
            self.linkify = False
        super().__init__(*args, **kwargs)

    def render_translated_text(self, value, record):  # override method in WordTable
        # if the user is logged on, then it may change the translation of a word.
        # compute the reverse of the word based on the word id in record and format a link.
        # if simply user, just display the black word.
        if self.linkify:
            args = f"{record['id']}"
            address = reverse('admin:elo_word_change', args=[args])
            return format_html(f"""<a href="{address}">{value}</a>""")
        else:
            return value

    source_text = tables.Column()
    translated_text = tables.Column()
    mp3 = tables.Column(verbose_name="Audio")


class PracticeTable(WordTable):
    """
    private practice table for selected words.
    """

    class Meta(SuperMeta):
        attrs = {
            "class": "table table-sm table-hover table-striped",
            "id": "practice-table",
            "thead": {
                "class": "thead-light"
            },
            "tbody": {
                "id": "practice-body"
            },
            "td": {
                "class": "practice-td"
            }
        }
        empty_text = ""

    def render_source_text(self, value, record):  # override method in WordTable
        return format_html(
            f"""
            <a href="#">{value}</a>"""
        )

    # other columns are inherited from WordTable
    translated_text = tables.Column()
