import django
from django.forms import widgets
from django.db import models


class SimpleTextField(models.TextField):
    """Just like a normal TextField, but uses a widget that is optimized
    for single-line input"""

    def formfield(self, **kwargs):
        return super().formfield(widget=widgets.TextInput)


class Language(models.Model):
    name = SimpleTextField(db_column='NAME', blank=True, null=True)
    adddate = models.DateTimeField(db_column='ADDDATE', blank=True, null=True)
    mtdate = models.DateTimeField(db_column='MTDATE', blank=True, null=True)
    mtanr = SimpleTextField(db_column='MTANR', blank=True, null=True)
    flag_url = SimpleTextField(db_column='FLAG_URL', blank=True, null=True)
    comments = models.TextField(db_column='COMMENTS', blank=True, null=True)

    class Meta:
        db_table = 'ELOQUENT_LANGUAGE'

    def __str__(self):
        return self.name


class Dialect(models.Model):
    # Field name made lowercase. This field type is a guess.
    language = models.ForeignKey(Language, on_delete=models.CASCADE)
    # Field name made lowercase. This field type is a guess.
    name = SimpleTextField(db_column='NAME', blank=True, null=True)
    # Field name made lowercase.
    adddate = models.DateTimeField(db_column='ADDDATE', blank=True, null=True)
    # Field name made lowercase.
    mtdate = models.DateTimeField(db_column='MTDATE', blank=True, null=True)
    # Field name made lowercase. This field type is a guess.
    mtanr = SimpleTextField(db_column='MTANR', blank=True, null=True)

    class Meta:
        db_table = 'ELOQUENT_DIALECT'

    def __str__(self):
        return f"{self.language}-{self.name}"


class Domain(models.Model):
    # Field name made lowercase. This field type is a guess.
    language = models.ForeignKey(Language, on_delete=models.CASCADE)
    # Field name made lowercase. This field type is a guess.
    name = SimpleTextField(db_column='NAME', blank=True, null=True)
    # Field name made lowercase.
    mtdate = models.DateTimeField(db_column='MTDATE', blank=True, null=True)
    # Field name made lowercase. This field type is a guess.
    mtanr = SimpleTextField(db_column='MTANR', blank=True, null=True)
    # Field name made lowercase.
    adddate = models.DateTimeField(db_column='ADDDATE', blank=True, null=True)

    class Meta:
        db_table = 'ELOQUENT_DOMAIN'

    def __str__(self):
        return f'{self.name}'


class Mp3(models.Model):
    name = SimpleTextField(db_column='NAME', blank=True, null=True)
    mime_type = SimpleTextField(db_column='MIME_TYPE', blank=True, null=True)
    doc_size = SimpleTextField(db_column='DOC_SIZE', blank=True, null=True)
    dad_charset = SimpleTextField(db_column='DAD_CHARSET', blank=True, null=True)
    content_type = SimpleTextField(db_column='CONTENT_TYPE', blank=True, null=True)
    content = models.TextField(db_column='CONTENT', blank=True, null=True)
    blob_content = models.BinaryField(db_column='BLOB_CONTENT', blank=True, null=True)
    addate = models.DateTimeField(db_column='ADDATE', blank=True, null=True)
    mtdate = models.DateTimeField(db_column='MTDATE', blank=True, null=True)
    # name made lowercase.
    mtanr = models.TextField(db_column='MTANR', blank=True, null=True)

    class Meta:
        db_table = 'ELOQUENT_MP3'

    def __str__(self):
        return self.name


class Role(models.Model):
    name = SimpleTextField(db_column='NAME')
    adddate = models.DateTimeField(auto_now_add=True, db_column='ADDDATE')
    mtdate = models.DateTimeField(db_column='MTDATE', null=True, blank=True)
    mtanr = models.IntegerField(db_column='MTANR')

    class Meta:
        db_table = 'ELOQUENT_ROLE'

    def __str__(self):
        return f'role-{self.name}'


class EloquentUser(django.contrib.auth.models.User):
    anr = SimpleTextField(db_column='ANR')
    role = models.ForeignKey(Role, on_delete=models.CASCADE)
    active = models.BooleanField(db_column='ACTIVE')
    name = SimpleTextField(db_column='NAME')
    sortname = SimpleTextField(db_column='SORTNAME')
    lastlogin = models.DateTimeField(db_column='LASTLOGIN', null=True, blank=True)
    adddate = models.DateTimeField(db_column='ADDDATE', auto_now_add=True)
    mtdate = models.DateTimeField(db_column='MTDATE', null=True, blank=True)
    telno = SimpleTextField(db_column='TELNO', null=True, blank=True)
    attempt = models.IntegerField(db_column='ATTEMPT', default=0, null=True)

    class Meta:
        db_table = 'ELOQUENT_USER'

    def __str__(self):
        return f'user-{self.name}'


class Word(models.Model):
    source_text = SimpleTextField()
    source_language = models.ForeignKey(
        Language, verbose_name="Source language", related_name="source_language",
        on_delete=models.CASCADE)
    dialect = models.ForeignKey(Dialect, verbose_name="Pronunciation", on_delete=models.CASCADE)
    domain = models.ForeignKey(Domain, on_delete=models.CASCADE, default=1)
    translated_text = SimpleTextField()
    target_language = models.ForeignKey(
        Language, verbose_name="Translated language", related_name="target_language",
        on_delete=models.CASCADE)
    mp3_number = models.IntegerField()
    mp3 = models.FileField(upload_to='mp3', blank=True, null=True)
    adddate = models.DateTimeField(auto_now_add=True)
    mtdate = models.DateTimeField(null=True, blank=True)

    class Meta:
        db_table = 'ELOQUENT_WORD'

    def get_absolute_url(self):
        # TODO: replace this with the word translate form
        return f'admin/elo/word/{self.id}'

    def __str__(self):
        return f'{self.source_text}'


class Translation(models.Model):
    word = models.ForeignKey(Word, related_name="translation_word",
                             on_delete=models.CASCADE)
    lang = models.ForeignKey(Language, related_name="lang",
                             on_delete=models.CASCADE,
                             default=1)
    text = SimpleTextField(blank=True, null=True)

    class Meta:
        db_table = 'ELOQUENT_TRANSLATION'

    def __str__(self):
        return f'{self.word}->{self.lang}'


class ExampleUsage(models.Model):
    word = models.ForeignKey(Word, related_name="example_word",
                             on_delete=models.CASCADE)
    example = models.TextField(blank=True, null=True)

    class Meta:
        db_table = 'ELOQUENT_EXAMPLE'

    def __str__(self):
        return f'{self.word}-{self.example}'


class Synonym(models.Model):
    word = models.ForeignKey(Word, related_name="synonym_word",
                             on_delete=models.CASCADE)
    synonym = SimpleTextField(blank=True, null=True)

    class Meta:
        db_table = 'ELOQUENT_SYNONYM'

    def __str__(self):
        return f'{self.word}-{self.synonym}'


class Practice(models.Model):
    word = models.ForeignKey(Word, on_delete=models.CASCADE)
    ip = models.TextField()

    class Meta:
        db_table = 'PRACTICE'

    def __str__(self):
        return f"{self.ip}-{self.word}"


class WordDialect(models.Model):
    word = models.ForeignKey(Word, on_delete=models.CASCADE)
    dialect = models.ForeignKey(Dialect, on_delete=models.CASCADE)
    mp3 = models.FileField(upload_to='mp3', blank=True, null=True)

    class Meta:
        db_table = 'WORDDIALECT'

    def __str__(self):
        return f"{self.word}-{self.dialect}"
