from django.core.handlers.wsgi import WSGIRequest
from django.db.models.functions import Lower
from django.http import HttpResponse
from django.shortcuts import redirect
from django_tables2 import SingleTableView, RequestConfig, Column

from . import logger, Session
from .filters import WordFilter
from .forms import WordFilterFormHelper
from .models import Word, Practice, Synonym, ExampleUsage
from .tables import WordTable, PracticeTable


class PagedFilteredTableView(SingleTableView):
    """
    Class to incorporate django-tables, crispy and django-filter
    """
    filter_class = None
    formhelper_class = None
    context_filter_name = 'filter'
    practice_table_class = None
    session = Session(practice_list=[], subset='a', synonyms=False,
                      examples=False, page=1)

    def set_page(self, page: int):
        self.session.page = page
        self.request.session['page'] = page

    def set_practice_page(self, page: int):
        self.session.practice_page = page
        self.request.session['p-page'] = page

    def set_subset(self, subset: str):
        self.session.subset = subset
        self.request.session['subset'] = subset

    def get_queryset(self, **kwargs):
        """
        get the initial query set ordered, and fix the crispy form and the filters
        :param kwargs:
        :return: a query set containing all words. filters can restrict the
        selection, see filters.py.
        """
        qs = super().get_queryset().order_by('source_language', Lower('source_text'))
        if PagedFilteredTableView.session.subset is not None and len(
            PagedFilteredTableView.session.subset) == 1:
            qs = qs.filter(source_text__istartswith=PagedFilteredTableView.session.subset)

        self.filter = self.filter_class(self.request.GET, queryset=qs)
        self.filter.form.helper = self.formhelper_class(
            subset=PagedFilteredTableView.session.subset)
        return self.filter.qs

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.ip = None

    def get_table(self, **kwargs):
        """
        get the table and return a paginated table.
        :param kwargs:
        :return:
        """
        columns = list()
        if PagedFilteredTableView.session.synonyms:
            columns.append(('synonym', Column()))

        if PagedFilteredTableView.session.examples:
            columns.append(('example', Column()))

        table = self.table_class(
            self.get_table_data(),
            extra_columns=columns,
            issuper=not self.request.user.is_anonymous
        )

        # compute the page
        if 'practice_values' in self.request.GET and \
            self.request.GET['practice_values'] and \
            'page' not in self.request.GET:
            page = self.request.session['page']
        else:
            page = self.request.GET.get('page', 1)
            self.set_page(page)

        RequestConfig(
            self.request,
            paginate={
                'page': page,
                'per_page': self.paginate_by
            }
        ).configure(table)

        return table

    def get_context_data(self, **kwargs):
        """
        add the filter class to the context. used by the html template.
        :param kwargs:
        :return:
        """
        context = super().get_context_data()
        context['practice'] = PracticeTable(
            data=Word.objects.filter(
                id__in=self.session.practice_list
            ).order_by(Lower("source_text")),
            prefix='p-'
        )
        context[self.context_filter_name] = self.filter
        context['subset'] = self.session.subset

        # Paginate the practice selection list (using prefix 'p-')
        practice_page = self.request.GET.get('p-page', 1)
        self.set_practice_page(practice_page)

        RequestConfig(
            self.request,
            paginate={
                'page': practice_page,
                'per_page': self.paginate_by
            }
        ).configure(context['practice'])

        return context

    def get(self, request: WSGIRequest, *args, **kwargs) -> HttpResponse:
        """
        Override default get to process filter and other submit buttons
        intercept any special actions to presentation of the word list.
        :param request:
        :return: HttpResponse
        """

        self.ip = request.META.get("REMOTE_ADDR")  # this ip is used to store
        self.request.session.setdefault('page', 1)
        self.request.session.setdefault('subset', 'a')
        self.session.practice_list = self.load_practice_words()
        # if page not sent in request, use the previous
        self.session.page = self.request.session['page']
        self.session.subset = self.request.session['subset']

        if 'page' in request.GET:
            self.set_page(request.GET.get("page"))

        if any([let in request.GET for let in 'abcdefghijklmnopqrstuvwxyz']):
            for let in request.GET:
                if len(let) == 1:  # really pressed a letter button?
                    self.set_subset(let)
                    self.set_page(1)

        if 'source_text__icontains' in request.GET and request.GET['source_text__icontains']:
            self.session.subset = None

        practice_set = set(self.session.practice_list)
        self.practice_table = PracticeTable(
            data=Word.objects.filter(id__in=practice_set).order_by("source_text"))

        # if "synonyms" in request.GET:
        #    self.session.synonyms = not self.session.synonyms
        #    # add synonyms column to table

        if "addword" in request.GET:
            return redirect("admin/elo/word")
            # go to admin/Word

        # if "example" in request.GET:
        #    self.session.examples = not self.session.examples
        # add example column to table

        self.make_collection_changes(self.request.GET)

        return super().get(request, *args, **kwargs)

    def delete_practice_words(self):
        pass

    def make_collection_changes(self, g: dict):
        pass

    def get_absolute_url(self):
        # TODO: replace this with the word translate form
        return f'admin/elo/word/{self.id}'

    def load_practice_words(self) -> list:
        pass


class WordTableView(PagedFilteredTableView):
    """
    Sub class to display words
    """
    model = Word
    table_class = WordTable
    practice_table_class = PracticeTable
    template_name = 'elo/word.html'
    paginate_by = 10
    filter_class = WordFilter
    formhelper_class = WordFilterFormHelper

    def keep_practice_words(self):
        """Store practice words in database"""

        # Clear practice list in session on 'Delete All'
        if self.request.GET.get('practice_values') == 'all':
            self.session.practice_list.clear()

        # first remove deleted words
        # so this would delete anything in the table which is not in the practice list
        Practice.objects.filter(ip=self.ip).exclude(word_id__in=self.session.practice_list).delete()

        # add records one by one from the include list
        for word in self.session.practice_list:
            if not Practice.objects.filter(word_id=word, ip=self.ip).exists():
                practice = Practice(word_id=word, ip=self.ip)
                practice.save()

    def load_practice_words(self) -> list:
        """load all practice words belonging to the current ip"""
        return Practice.objects.filter(
            ip=self.ip).values_list("word_id", flat=True).order_by(Lower("word__source_text"))

    def delete_practice_words(self):
        Practice.objects.filter(ip=self.ip).delete()

    def make_collection_changes(self, g: dict):
        """
        make changes to the practice table based on hidden input practice_values
        :return: None
        """
        words = g.get('practice_values', '').split(',')
        try:
            words = list(map(int, words))
        except ValueError:
            words = list()

        words_add = {w for w in words if w > 0}
        words_remove = {-w for w in words if w < 0}
        practice_set = set(self.session.practice_list)
        practice_set = practice_set.union(words_add)
        practice_set = practice_set.difference(words_remove)
        self.session.practice_list = list(practice_set)  # session cannot contain sets
        self.keep_practice_words()

    def get_table_data(self):
        data = self.get_queryset().values()

        for record in data:
            if self.session.synonyms:
                record['synonym'] = ', '.join(Synonym.objects.filter(
                    word__source_text=record['source_text']).values_list(
                    'synonym', flat=True).order_by('synonym'))
            if self.session.examples:
                record['example'] = ', '.join(ExampleUsage.objects.filter(
                    word__source_text=record['source_text']).values_list(
                    'example', flat=True).order_by('example'))
        return data
