/*
 * This file is part of mpv.
 *
 * mpv is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * mpv is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with mpv.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <math.h>

#include "audio/aframe.h"
#include "audio/format.h"
#include "filters/f_autoconvert.h"
#include "filters/filter_internal.h"
#include "filters/user_filters.h"
#include "options/m_option.h"


/* This is xoroshiro64* 1.0, our best and fastest 32-bit small-state
   generator for 32-bit floating-point numbers. We suggest to use its
   upper bits for floating-point generation, as it is slightly faster than
   xoroshiro64**. It passes all tests we are aware of except for linearity
   tests, as the lowest six bits have low linear complexity, so if low
   linear complexity is not considered an issue (as it is usually the
   case) it can be used to generate 32-bit outputs, too.

   We suggest to use a sign test to extract a random Boolean value, and
   right shifts to extract subsets of bits.

   The state must be seeded so that it is not everywhere zero.

   Written in 2016 by David Blackman and Sebastiano Vigna (vigna@acm.org)

   To the extent possible under law, the author has dedicated all copyright
   and related and neighboring rights to this software to the public domain
   worldwide. This software is distributed without any warranty.

   See <http://creativecommons.org/publicdomain/zero/1.0/>. */


static inline uint32_t rotl32(const uint32_t x, int k) {
	return (x << k) | (x >> (32 - k));
}

static uint32_t xoroshiro64_state[2] = { 0xD50058E7, 0x4C01600C };

static uint32_t xoroshiro64(void) {
	const uint32_t s0 = xoroshiro64_state[0];
	const uint32_t s1 = xoroshiro64_state[1];
	const uint32_t s2 = s0 ^ s1;

	xoroshiro64_state[0] = rotl32(s0, 26) ^ s2 ^ (s2 << 9); // a, b
	xoroshiro64_state[1] = rotl32(s2, 13); // c

	return s0 * UINT32_C(0x9E3779BB);
}


/* This is xoroshiro128+ 1.0, our best and fastest small-state generator
   for floating-point numbers. We suggest to use its upper bits for
   floating-point generation, as it is slightly faster than
   xoroshiro128**. It passes all tests we are aware of except for the four
   lower bits, which might fail linearity tests (and just those), so if
   low linear complexity is not considered an issue (as it is usually the
   case) it can be used to generate 64-bit outputs, too; moreover, this
   generator has a very mild Hamming-weight dependency making our test
   (http://prng.di.unimi.it/hwd.php) fail after 5 TB of output; we believe
   this slight bias cannot affect any application. If you are concerned,
   use xoroshiro128++, xoroshiro128** or xoshiro256+.

   We suggest to use a sign test to extract a random Boolean value, and
   right shifts to extract subsets of bits.

   The state must be seeded so that it is not everywhere zero. If you have
   a 64-bit seed, we suggest to seed a splitmix64 generator and use its
   output to fill s.

   NOTE: the parameters (a=24, b=16, b=37) of this version give slightly
   better results in our test than the 2016 version (a=55, b=14, c=36).

   Written in 2016-2018 by David Blackman and Sebastiano Vigna (vigna@acm.org)

   To the extent possible under law, the author has dedicated all copyright
   and related and neighboring rights to this software to the public domain
   worldwide. This software is distributed without any warranty.

   See <http://creativecommons.org/publicdomain/zero/1.0/>. */

static inline uint64_t rotl64(const uint64_t x, int k) {
	return (x << k) | (x >> (64 - k));
}

static uint64_t xoroshiro128_state[2] = { 0x8BE7D1A8B6091D43, 0x98FA42FB3E3EDE75 };

static uint64_t xoroshiro128(void) {
	const uint64_t s0 = xoroshiro128_state[0];
	const uint64_t s1 = xoroshiro128_state[1];
	const uint64_t s2 = s0 ^ s1;

	xoroshiro128_state[0] = rotl64(s0, 24) ^ s2 ^ (s2 << 16); // a, b
	xoroshiro128_state[1] = rotl64(s2, 37); // c

	return s0 + s1;
}

struct af_nosilence_opts {
	int power;
};

struct af_nosilence_priv {
	struct af_nosilence_opts *opts;
};

static void process(struct mp_filter *filt) {
	if(!mp_pin_can_transfer_data(filt->ppins[1], filt->ppins[0]))
		return;

	struct mp_frame frame = mp_pin_out_read(filt->ppins[0]);

	struct af_nosilence_priv *priv = filt->priv;
	struct af_nosilence_opts *opts = priv->opts;
	int power = opts->power;

	switch(frame.type) {
		case MP_FRAME_AUDIO: {
			struct mp_aframe *aframe = frame.data;
			if(mp_aframe_get_size(aframe)) {
				uint8_t **data = mp_aframe_get_data_rw(aframe);
				int size = mp_aframe_get_size(aframe);
				int channels = mp_aframe_get_channels(aframe);
				int format = mp_aframe_get_format(aframe);
				switch(format) {
					case AF_FORMAT_U8: {
						int8_t *samples = data[0];
						for(int i = 0; i < size * channels; i++)
							if(samples[i] == 127 || samples[i] == 128)
								samples[i] = 127 + (xoroshiro64() >> 31);
						break;
					}

					case AF_FORMAT_U8P: {
						for(int c = 0; c < channels; c++) {
							uint8_t *samples = data[c];
							for(int i = 0; i < size; i++)
								if(samples[i] == 127 || samples[i] == 128)
									samples[i] = 127 + (xoroshiro64() >> 31);
						}
						break;
					}

					case AF_FORMAT_S16: {
						if(power == 0)
							power = 15;
						int16_t *samples = (int16_t *)data[0];
						for(int i = 0; i < size * channels; i++) {
							int32_t s = (int32_t)samples[i] + ((int16_t)(xoroshiro64() >> 16) >> power);
							samples[i] = MPCLAMP(s, INT16_MIN, INT16_MAX);
						}
						break;
					}

					case AF_FORMAT_S16P: {
						if(power == 0)
							power = 15;
						for(int c = 0; c < channels; c++) {
							int16_t *samples = (int16_t *)data[c];
							for(int i = 0; i < size; i++) {
								int32_t s = (int32_t)samples[i] + ((int16_t)(xoroshiro64() >> 16) >> power);
								samples[i] = MPCLAMP(s, INT16_MIN, INT16_MAX);
							}
						}
						break;
					}

					case AF_FORMAT_S32: {
						if(power == 0)
							power = 23;
						int32_t *samples = (int32_t *)data[0];
						for(int i = 0; i < size * channels; i++) {
							int64_t s = (int64_t)samples[i] + ((int32_t)xoroshiro64() >> power);
							samples[i] = MPCLAMP(s, INT32_MIN, INT32_MAX);
						}
						break;
					}

					case AF_FORMAT_S32P: {
						if(power == 0)
							power = 23;
						for(int c = 0; c < channels; c++) {
							int32_t *samples = (int32_t *)data[c];
							for(int i = 0; i < size; i++) {
								int64_t s = (int64_t)samples[i] + ((int32_t)xoroshiro64() >> power);
								samples[i] = MPCLAMP(s, INT32_MIN, INT32_MAX);
							}
						}
						break;
					}

					case AF_FORMAT_FLOAT: {
						if(power == 0)
							power = 23;
						float *samples = (float *)data[0];
						for(int i = 0; i < size * channels; i++)
							samples[i] += ldexpf((xoroshiro64() >> 8) * 0x2.0p-24f - 1.0f, -power);
						break;
					}

					case AF_FORMAT_FLOATP: {
						if(power == 0)
							power = 23;
						for(int c = 0; c < channels; c++) {
							float *samples = (float *)data[c];
							for(int i = 0; i < size; i++)
								samples[i] += ldexpf((xoroshiro64() >> 8) * 0x2.0p-24f - 1.0f, -power);
						}
						break;
					}

					case AF_FORMAT_DOUBLE: {
						if(power == 0)
							power = 23;
						double *samples = (double *)data[0];
						for(int i = 0; i < size * channels; i++)
							samples[i] += ldexp((xoroshiro128() >> 11) * 0x2.0p-53 - 1.0, -power);
						break;
					}

					case AF_FORMAT_DOUBLEP: {
						if(power == 0)
							power = 23;
						for(int c = 0; c < channels; c++) {
							double *samples = (double *)data[c];
							for(int i = 0; i < size; i++)
								samples[i] += ldexp((xoroshiro128() >> 11) * 0x2.0p-53 - 1.0, -power);
						}
						break;
					}

					default:
						MP_ERR(filt, "unsupported audio format %s (%d)\n", af_fmt_to_str(format), format);
						mp_filter_internal_mark_failed(filt);
				}
			}
			break;
		}
		case MP_FRAME_EOF:
			break;
		default:
			MP_ERR(filt, "audio frame expected\n");
			mp_filter_internal_mark_failed(filt);
	}

	mp_pin_in_write(filt->ppins[1], frame);
}

static const struct mp_filter_info af_nosilence_filter = {
	.name = "nosilence",
	.priv_size = sizeof(struct af_nosilence_priv),
	.process = process,
};

static struct mp_filter *af_nosilence_create(struct mp_filter *parent, void *options) {
	struct mp_filter *filt = mp_filter_create(parent, &af_nosilence_filter);
	if (filt) {
		struct af_nosilence_priv *priv = filt->priv;
		priv->opts = talloc_steal(priv, options);

		mp_filter_add_pin(filt, MP_PIN_IN, "in");
		mp_filter_add_pin(filt, MP_PIN_OUT, "out");
	} else {
		talloc_free(options);
	}

	return filt;
}

#define OPT_BASE_STRUCT struct af_nosilence_opts

const struct mp_user_filter_entry af_nosilence = {
	.desc = {
		.name = "nosilence",
		.description = "Make sure the stream is never completely silent",
		.priv_size = sizeof(struct af_nosilence_opts),
		.options = (const struct m_option[]) {
			{"power", OPT_INT(power), OPTDEF_INT(0), M_RANGE(7, 64)},
			{0}
		},
	},
	.create = af_nosilence_create,
};
