package Displayentry;
use strict;
use warnings  FATAL=> 'all';

our @ISA=('Baseobject' );
use XML::LibXML;
use base qw(XML::LibXML::Document);
use utf8;

use Baseobject;
use Exporter;
use Babelfish;
use Ldapsearch;

sub new {
	my $pkg = shift;
	my $self = bless (new Baseobject(@_),$pkg);

	$self->needs([qw(displaytypes multiple entry configTree
		      multhash babelfish outTree orgTree hierarchy limit ldapSearch)]);

	$self->allows([qw(attribAccess)]);

	$self->fields(@_);
	$self->checkneeds();
	$self->{fields}->{cfhash} = $self->{fields}->{ldapSearch}->{fields}->{cfhash};
	return $self if $self->error();

	$self->{isOrganization} = grep /^organizationalunit$/io,  $self->get_value('objectclass');

	#gather language dependend attribs in a hash
	map {$self->{lanDepAttr}->{$_->nodeName} = 1} $self->{fields}->{configTree}->{root}->findnodes("/top/language_dependend_attribs/*");

	$self->{dn} = $self->get_value('dn') ;
	$self->gatherCategories();
	$self->growTree();
	$self;
}

sub makeHumanCategories {
	my $self = shift;
	my @statusList = $self->get_value('organizationalStatus;lang-en');

	foreach my $humanType (qw(staff pnil student guest)) {
		my $regexp = $self->{fields}->{cfhash}->{$humanType."Regexp"};
		if (grep (/$regexp/i, @statusList)){
			push (@{$self->{categories}},$humanType);
		}
	}
}

sub gatherCategories {
	my $self = shift;
	my $dn = $self->{dn};

	if ($self->{isOrganization}){
#		warn "organizationalunit found: $dn";
		my $entry = $self->{fields}->{entry};
		my $orgID = $entry->get_value('orgID');

		if ($orgID) {
			my $h = $self->{fields}->{ldapSearch}->decodeOrgname($orgID);

			my $name = $self->{fields}->{entry}->get_value('description') ;
			my $depth = $h->{'depth'};

			if ($depth <= 1){
#				warn "toporganisation: $name";
				push (@{$self->{categories}}, 'toporganisation');
			} elsif ($depth > 1) {
#				warn "suborganisation: $name";
				push (@{$self->{categories}}, 'suborganisation');
			}
			return;
		} else {
			warn "No orgID found for: $dn";
			push (@{$self->{categories}}, 'toporganisation');
		}
	}
	my @var = $self->get_value('organizationalStatus;lang-en');
	$self->makeHumanCategories();

	# is this used at all?
	if (grep /^domainrelatedobject$/io, $self->get_value('objectclass')){
		push ( @{$self->{categories}}, 'university');
	} else {

		if (! grep /^uvtUser$/io, $self->get_value('objectclass'))	{
			push ( @{$self->{categories}},'mailalias');
		} elsif (! exists ($self->{categories})) {
			push ( @{$self->{categories}},'etranger');
		}
	}
	#  warn $self->inspect($self->{categories});
}

sub growTree {
	my $self = shift;
	my $dn = $self->{dn} ;
	my $name = $dn;

	$name = $self->{fields}->{entry}->get_value('description') if $self->{isOrganization};
	$name =~ s/,.*//o if $name;
	$self->{multiple} = $self->{fields}->{multiple} == 1? 'single' : 'multiple';

	# Toon de lijst met bijvoorbeeld suborganisaties
	foreach my $category (@{$self->{categories}}) {
		if ($self->{fields}->{limit}) {
            $self->debug("limit: $self->{fields}->{limit}, category: $category");
			next unless $category =~ /$self->{fields}->{limit}/o;
		}
		my $entryel = $self->output($self->{fields}->{outTree}->{apptop},'entry');
		$self->output($entryel,'type', $self->{multiple});

		my $title = $self->{fields}->{babelfish}->translate($category);
		my $categoryel = $self->output ($entryel ,$category,'-', { 'title'=>"\u$title" });
 
 		$self->output($categoryel,'name',$name);

		if ($category =~ /organisation$/o and  exists ($self->{fields}->{multhash}->{$category})) {
			if ($self->{fields}->{multhash}->{$category} >1) {
				$self->{multiple} = 'multiple';
			} else {
				$self->{multiple} = 'single';
			}
		}

		$self->output($categoryel,'type',$self->{multiple});

		my $path = "/top/categories/$category/$self->{multiple}/*";
		my @list = $self->{fields}->{configTree}->{root}->findnodes($path);

		# walk through configuation nodes, read attribs
		foreach my $el (@list) {
			my $attr = $el->nodeName();
			if (exists($self->{fields}->{attribAccess}->{$attr})) {
				next unless $self->{fields}->{attribAccess}->{$attr};# may contain 'limitedAccess or 'Granted''
			}

			my ($confNode) = $self->{fields}->{configTree}->{root}->findnodes("/top/categories/$category/$self->{multiple}/$attr");
			if ($confNode->getAttribute('skipifempty')) {
				my @list = $self->get_value($attr);
				next unless @list;
			}

			if ($confNode->getAttribute('skipifstudent')) {
				next if $category eq 'student';
			}

			unless (exists ($self->{mononoclick}->{$attr})) {
				$self->{mononoclick}->{$attr} = 1;
				my $mononoclick = $confNode->getAttribute('mononoclick');
				if (defined($mononoclick)) {
					my $ldap = $self->{fields}->{ldapSearch};
					my ($currentcontent) = $self->get_value($attr);
					if (defined($currentcontent)) {
						my $msg = $ldap->ldapsearch("($attr=$currentcontent)");
						if ($msg->code) {
							warn "ldap search failed: ", $msg->error;
						} else {
							my @entries = $msg->entries();
							$self->{mononoclick}->{$attr} = 0 if @entries <= 1;
						}
					}
				}
			}

			unless ($self->{fields}->{displaytypes}->{$attr}) {
				warn "attribute not defined in ldapattribs: $attr";
				return;
			}

			my $bf = $self->{fields}->{babelfish}->translate($attr,['class']);
			my $curel = $self->output($categoryel, 'attr', $attr, {'class' => $bf->{class}});

			$self->output($curel,'name',"\u$bf->{value}");
			$self->addvalues($category, $attr, $curel, $confNode);
		}
		last if $self->{multiple} eq 'single';
	}
}


use Carp;

sub hierarchySort {
	# Hier kan nog een sort komen gebasseerd op hierarchie ipv alfabet
	my $self = shift;
	my $ous = shift;
#	warn $self->inspect($self->{fields}->{ldapSearch}->{hierarchy});
 	return sort(@$ous);
}


sub addvalues {
	my ($self, $category, $attr, $curel, $confNode) = @_;
	my $type = $self->{fields}->{displaytypes}->{$attr};
	
	my @regexpnodes = $self->{fields}->{configTree}->{root}->findnodes("/top/categories/$category/$self->{multiple}/$attr/regexp");
	my @hierarchynodes = $self->{fields}->{configTree}->{root}->findnodes("/top/categories/$category/$self->{multiple}/$attr/hierarchy");
	my $href = $confNode->getAttribute('href');
	#warn "attr: $attr href: $href" if ($href);
	my $suppresstophref = $confNode->getAttribute('suppresstophref');
	$suppresstophref = 0 unless defined ($suppresstophref);

	my $overruleAttributeType = $confNode->getAttribute('attributeType');
	$type = $overruleAttributeType if defined($overruleAttributeType);

	my $regexpptr = \@regexpnodes;

	my @valuelist;
	if ($self->{isOrganization} and $attr eq 'ou' ){
		@valuelist = $self->{fields}->{entry}->get_value('description');
	} else {
		@valuelist = $self->get_value($attr);
	}

	# format values in output
	if (my $format = $confNode->getAttribute('format')) {
		my $h;
		my $i = 0;
		my $delimitor;
		my $seperator;
		unless ($delimitor = $confNode->getAttribute('fdelimitor')) {
			$delimitor = '-';
		}

		unless ($seperator = $confNode->getAttribute('fseperator')) {
			$seperator = $delimitor;
		}

		map {$h->{$_} = $i++} split($delimitor, $format);
		my $show;
		unless ($show = $confNode->getAttribute('fshow')) {
			$show = $format;
		}
		my @showlist = split($delimitor, $show);
		my $orgvalue = $valuelist[0];
		my @elements = split($delimitor, $orgvalue);

		my @newvalue;
		foreach my $el (@showlist) {
			if (exists ($h->{$el})) {
				push (@newvalue, $elements[$h->{$el}]) if $elements[$h->{$el}];

			} else {
				warn "Error: element: \"$el\" from 'fshow' not in 'format':", keys(%$h);
			}
		}
		$valuelist[0] = join($seperator, @newvalue) if @newvalue;
	}


	$self->output($curel, 'value', '-') unless @valuelist;
	my $replacer = $self->getReplacerWithinValue($category, $attr);
	my @sortedValuelist;

	if (@hierarchynodes) {
		foreach my $value (@valuelist){
			$self->{fields}->{ldapSearch}->addhierarchy($value, @valuelist) ;
		}
#		$self->{fields}->{ldapSearch}->finalizehierarchy() ;
		@sortedValuelist = $self->hierarchySort(\@valuelist);
	} else {
		@sortedValuelist = sort(@valuelist);
	}


	my $conditionalAction = $confNode->getAttribute('conditionalAction') // 0;
	my $access = $self->{fields}->{attribAccess}->{$attr} // '';

	my $proceed = 1;
	$proceed = 0 if $conditionalAction && $access eq 'limitedAccess';

	my $conditionalFilter = $confNode->getAttribute('conditionalFilter');

	foreach my $value (@sortedValuelist) {
		chomp $value;
		if ($replacer) {
			my ($old, $new) = split(/\//, $replacer);
			$value =~ s/$old/$new/g;
		}

        # expliciete href overschrijft de automatisch gegenereerd link
        #
		if ($href && $proceed) {
			my $title = $confNode->getAttribute('title') || '';

			if ($href =~ /\$/o) {
				my @list = $href =~ /\$([\w\{\}]+)/gos;
				my $subst;

				foreach my $word (@list) {
					($subst->{$word}) = $self->get_value($word);
					$subst->{$word} =~ s/(\W)/\\$1/go;
				}
				$self->expandvars(\$href,$subst);
			}
			
			my $onclick = $confNode->getAttribute('onclick') || '';
			$self->output ($curel,'value',"$value", {'href'=>$href, 'onclick'=>$onclick, 'title'=>$title });
			next;
		}


		# FILTER
		if ($proceed &&
			($type eq 'filter' or ($conditionalFilter and $value =~ /$conditionalFilter/i))
			and ( $self->{mononoclick}->{$attr})
			or ($proceed && $type eq 'multifilter' && $self->{multiple} eq 'multiple' )
			)
		{
			$self->generatefilter($category, $curel, $attr, $value, $suppresstophref);
		}
		# URL
		elsif ($type eq 'url') {
			#
			# ldap bevat iets als:  labeledURI: http://www.uvt.nl/webwijs/show.html?anr=780251
			# beetje onzin dat dit in ldap staat, het is afgeleide info, leidt het nu tweetalig af
			#
			if ($value =~ /webwijs/) {
				my ($anr) = $value =~ /anr=(\d+)/;
				my $u = $self->{fields}->{babelfish}->translate('webwijsUrl');
				$value = "$u$anr";
			}

			my $h = $self->{fields}->{babelfish}->translate($attr, {'title' => 1});
			$self->output($curel, 'value', $h->{title}, {'href' => $value});
		}
		# MAILTO
		elsif ($type eq 'mailto'){
			$self->output($curel,'value',$value, {
							   'href'=>"mailto:$value",
							   'mailto'=>1,
						   }) ;
		} else {
			$self->output($curel,'value',$value);
		}
	}
}


sub getReplacerWithinValue {
	my ($self, $category, $attr) = @_;
	my @listNodes = $self->{fields}->{configTree}->{root}->findnodes("/top/categories/$category/$self->{multiple}/$attr");
	if (@listNodes) {
		 return $listNodes[0]->getAttribute('replace');
	}
	return undef;
}

sub generatefilter {
	my ($self, $category, $curel, $attr, $value, $suppresstophref) = @_;
	my @filter = $self->{fields}->{configTree}->{root}->findnodes("/top/categories/$category/$self->{multiple}/$attr/filter");
	my @listNodes = $self->{fields}->{configTree}->{root}->findnodes("/top/categories/$category/$self->{multiple}/$attr");
	my $refencoded;
	my $href;
	my $h;
	my $orgvalue = $value;
	my $searchContext;
	my $overrideValue = $value;
	my $noprint = '';

	if (@listNodes) {
		my $overrideAttr = $listNodes[0]->getAttribute('overrideAttr');
		if (defined $overrideAttr) {
			$attr = $overrideAttr;
			($overrideValue) = $self->get_value($attr);
		}

		my $replace = $listNodes[0]->getAttribute('replaceWithinFilter');
		if ($replace) {
			my ($org, $new) = split (/\//,$replace);
			$overrideValue =~ s/$org/$new/g;
		}
	
		my $probeNoprint = $listNodes[0]->getAttribute('noprint');
		if (defined($probeNoprint)) {
			$noprint = '_noprint';
		}
	}

	my $class = "value$noprint";
	if ($attr eq 'ou') {
		$class = "ou_value$noprint";

		my $x = Ldapsearch::ldapsearchencode($orgvalue);
		$searchContext = urlencode($orgvalue);
		$refencoded = "(&(objectclass=organizationalunit)($attr=$x))";
	} else {
		$searchContext = urlencode($self->{fields}->{ldapSearch}->{fields}->{cgiparams}->{searchContext});
		$refencoded = "$attr=" . Ldapsearch::ldapsearchencode($overrideValue);
	}

	$self->{fields}->{outTree}->{doc}->createTextNode($orgvalue);
	$href = "?searchContext=$searchContext&ldapfilter=" . urlencode(bracket($refencoded));

	unless (@filter) {
		# singular filter
		$self->output ($curel, 'value', $value,
					   {
						   class => $class,
						   href => $href,
					   });
	} else {
		# additional ldapfilters
		my $ldapencoded = Ldapsearch::ldapsearchencode($value);
		my $deeper;

		unless ($suppresstophref) {
			$deeper = $self->output($curel, 'value', "$value",
								   {
									   href => $href,
									   class => $class,
									   deeper => $value,
								   }) ;
		} else {
			$deeper = $self->output($curel, 'value', $value,
								   {
									   class => $class,
									   deeper => $value,
								   });
		}

		foreach my $additionalFilter (@filter) {
			my $xtrafilter = $additionalFilter->getAttribute('filter');
			my $limit = $additionalFilter->getAttribute('limit') || '';

			# NB 'waarvoor die 'add'?
			my $add = $additionalFilter->getAttribute('add');
			$add = 1 unless defined($add);
			$add = 0 if $add eq 'false';

			my $xtravalue = $additionalFilter->textContent();
			my $basefilter = ("$attr=".Ldapsearch::ldapsearchencode($orgvalue));
			my $newfilter = $add?"(&($basefilter)($xtrafilter))":$xtrafilter;
			my $bf = $self->{fields}->{babelfish}->translate($xtravalue,['title', 'abbrev']);
			my $attrhash = {
				'href' => "?searchContext=$searchContext&$limit&ldapfilter=".urlencode(bracket($newfilter)),
				'notrans' => $xtravalue,
				'abbrev' => $bf->{abbrev},
			};

			# submenuutje voor ou's, images only
			$attrhash->{title} = $bf->{title};
			$self->output($deeper,'submenu',$bf->{value}, $attrhash ) ;
		}
	}
}

sub urlencode {
	my ($x) = @_ ;
	return "" if !defined ($x);

	$x = $_[0] ;
	$x =~ s/([^ \w])/sprintf("%%%02X",ord($1))/geo;
	$x =~ s/ /\+/go ;
	$x ;
}

sub output {
	my ($self, $ldapSearch, $el, $content, $attrhash) = @_;
	my $newel;

	$newel=$self->{fields}->{outTree}->{doc}->createElement($el);
	$ldapSearch->appendChild($newel);

	$newel->appendChild($self->{fields}->{outTree}->{doc}->createTextNode($content)) if defined $content;

	if (defined ($attrhash)) {
		foreach my $attr (keys (%$attrhash)) {
			$newel->setAttribute($attr,$attrhash->{$attr}) if defined $attrhash->{$attr};
		}
	}
	return $newel;
}

sub get_value {
	# returns a sorted LIST OF VALUES
	my ($self, $attr) = @_;
	my $langattrib;
	my @res;

	# PSEUDO attributes
	if ($attr eq 'dn') {
		my $dn = $self->{fields}->{entry}->dn();
		$dn =~ s/,.*//o;
		$dn =~ s/^.*=//o;
		return $dn;
	}

	if ($attr eq 'DN') {
	    return $self->{fields}->{entry}->dn();
	}

	if ($attr eq 'blockradius' or $attr eq 'showradius' ) {
	    my @uvtAuth = $self->getNormalizedValues('uvt-auth');
		my $normalAccess = $self->{fields}->{cfhash}->{radiusDefaultUvTAuthString};
	    my $normal = 1;
	    my $value = '-';

		if (@uvtAuth) {
			$normal = grep (/^$normalAccess/, @uvtAuth);
			$value = $self->{fields}->{cfhash}->{radiusShowUnblockedString};
			$value .= " $self->{fields}->{cfhash}->{radiusUnblockedString}" if $attr eq 'blockradius';
		}

		unless ($normal) {
			my $currentBlock = $self->{fields}->{cfhash}->{radiusBlockedUvTAuthString};
			my ($radius_revoked) = grep(/^$currentBlock/, @uvtAuth);
			if ($radius_revoked) {
				my ($date) = $radius_revoked =~ /at\s+(.+)/;
				my $subst; $subst->{date} = $date;
				$value = $self->{fields}->{cfhash}->{radiusShowBlockedString};
				$value .= " $self->{fields}->{cfhash}->{radiusBlockedString}" if $attr eq 'blockradius';
				$self->expandvars(\$value, $subst);
			}
		}
	    return $value;
	}

	# real attributes
	if ($attr =~/^;lang-/o) {
		return $self->getNormalizedValues($attr);
	}

	my @languagelist;

	if (exists $self->{lanDepAttr}->{$attr}) {
#		warn "lanDepAttr: $attr, lan:($self->{fields}->{babelfish}->{fields}->{language}  ";
		if ($self->{fields}->{babelfish}->{fields}->{language})    {
			push (@languagelist, "lang-$self->{fields}->{babelfish}->{fields}->{language}");
		}
	} else {
		push (@languagelist, 'lang-nl');
	}

	foreach $langattrib (@languagelist) {
		push (@res,$self->getNormalizedValues("$attr;$langattrib"));
	}

	push (@res, $self->getNormalizedValues($attr));
	my $h;  map {$h->{$_} = 1} @res;
	return (sort keys (%$h));
}

sub getNormalizedValues {
  my ($self,$attr) = @_;
  my @res;

  if ($attr eq 'street') {
	  map { push (@res, split(/\,/o)) } $self->{fields}->{entry}->get_value($attr);
  }  else  {
	  map { push (@res, split(/\$/o)) } $self->{fields}->{entry}->get_value($attr);
  }
  @res;
}


sub bracket {
  my $string=shift;

  return $string if $string =~ /^\s*\(/o;
  return "($string)";
}

42;

=head1 NAME

Hello - class to greet a user

=head1 METHODS

=head2 $hello->greet()

Prints a greeting.
