#! /usr/bin/perl

use strict;
use warnings FATAL => 'all';

package FilterAnalyzer;

use Net::LDAP::Filter;
use Scalar::Util qw(reftype);

sub new {
	my $class = shift;
	my $self = bless {}, $class;

	while(@_) {
		my $method = shift;
		my $argument = shift;
		$self->$method($argument);
	}

	return $self;
}

sub analyze_not {
	my ($self, $args) = @_;
	return $self->_analyze($args);
}

sub analyze_and {
	my ($self, $args) = @_;
	return [map { @{$self->_analyze($_)} } @$args];
}

sub analyze_or {
	my ($self, $args) = @_;
	return [map { @{$self->_analyze($_)} } @$args];
}

sub analyze_extensibleMatch {
	my ($self, $args) = @_;
	return [$args->{type}];
}

sub analyze_substrings {
	my ($self, $args) = @_;
	return [$args->{type}];
}

sub analyze_present {
	my ($self, $attribute) = @_;
	return [$attribute];
}

sub analyze_equalityMatch {
	my ($self, $args) = @_;
	return [$args->{attributeDesc}];
}

sub analyze_approxMatch {
	my ($self, $args) = @_;
	return [$args->{attributeDesc}];
}

sub analyze_greaterOrEqual {
	my ($self, $args) = @_;
	return [$args->{attributeDesc}];
}

sub analyze_lessOrEqual {
	my ($self, $args) = @_;
	return [$args->{attributeDesc}];
}

sub _analyze {
	my ($self, $filter) = @_;
	my ($op, $args) = %$filter;

	my $sub = "analyze_$op";
	return $self->$sub($args) if $self->can($sub);

	die "filter operator '$op' not supported by ".__PACKAGE__."\n";
}

sub analyze {
	my ($self, $filter) = @_;

	$filter = new Net::LDAP::Filter($filter) unless ref $filter;

	my $attrs = $self->_analyze($filter);

	my %uniq;
	@uniq{map { lc } @$attrs} = @$attrs;

	return [sort { lc($a) cmp lc($b) } values %uniq];
}

42;

#package main;
#
#use Data::Dumper;
#
#my $analyzer = new My::LDAP::FilterAnalyzer;
#
#print join(", ", @{$analyzer->analyze(join(' ', @ARGV))}), "\n";
